local MDP_Maple, TaylorShift1, BHL_Maple;
# MDP_Maple solves a Diophantine equation sigma[1]*U[1]+...sigma[r]*U[r] = ck mod p
# for sigma[rr], rr=1..r. where U[i]=prod(f0[rr],rr=1..r)/f0[i] 
# f0[1],...,f[r],and ck are given univariate polynomials.
# Input: f0 is an Array of r polynomials in Zp[x], 
#        ck is a polynomial in Zp[x], 
#        p is a prime, 
#        M is an Array of pre-computed products s.t. M[r]=1, M[r-1]=f0[r],..., M[1]=f0[r]*f0[r-1]...f0[2].
# Output: sigma, an Array of r polynomials s.t. sum(sigma[rr]*U[rr],rr=1..r) = ck mod p
# MDP_Maple uses Algorithm 7 in Garrett Paluck's MSc thesis (2019).
MDP_Maple := proc( f0::list(polynom), # list of r polynomials in Zp[x] 
                   ck::polynom, # polynomial in Zp[x] 
                   x::name, # variable name 
                   p::prime, # a prime
                   M::Array(polynom), # Pre-computed products
                   sigma::Array(polynom) ) # answer Array  
    local r::'integer':=numelems(f0), # number of polynomials in f0
          c0::'polynom':=ck, # intermediate polynomial, step 5
          s::'polynom', # intermediate polynomial
          t::'polynom', # intermediate polynomial
          q::'polynom', # intermediate polynomial
          i::'integer'; # loop variable
    if r <> numelems(sigma) then 
        error "size of the answer Array incorrect";
    elif r <> numelems(M) then
        error "size of the pre-computed products incorrect"; 
    end if;
    # Step 7-12:
    for i to r-1 do
  	Gcdex(M[i],f0[i],x,'s','t') mod p;
        sigma[i] := Rem(c0*s,f0[i],x,'q') mod p;
        c0 := Expand(c0*t+q*M[i]) mod p;
    end;
    sigma[r] := c0; # Step 13
    return;
end:

# TaylorShift1 computes the coefficient of (y-alpha)^k in f, where k is given,
# f is a given polynomial in y, i.e. f = ad*y^d + ... + a1*y + a0, 
# a0,...,ad can be integers mod p or polynomials (univariate or multivariate) mod p. 
# If k is not specified, the TaylorShift1 returns an Array of coefficients 
# of (y-alpha)^i, for i=0..dy, where dy = deg(f,y).  
TaylorShift1 := proc( f::polynom, # input polynomial
                      y::name,    # variable name of f
                      alpha::integer, # given integer
                      p::prime, # a prime
                      k::nonnegint )  # specfied power (optional)
    local g::polynom:=f, # intermediate polynomial
          dy::integer:=degree(f,y), # deg(f,y)
          A::Array(polynom), # answer array 
          i::integer, # loop variable
          r::polynom; # answer polynomial 
    if _params['k'] = NULL then # returns an Array of coeffs of (y-alpha)^i in f, i=0..dy
        if f = 0 then 
            return Array(0..0);
        end if; 
        A := Array(0..dy);
        for i from 0 to dy do
            g := Quo(g,y-alpha,y,'r') mod p;
            A[i] := r;
        end do;
        return A;
    else # returns coeff of (y-alpha)^k in f
        if f = 0 then 
            return 0; 
        end if;
        for i from 0 to k do 
            g := Quo(g,y-alpha,y,'r') mod p; 
        end do;
        return r;
    end if;
end:

# BHL_Maple: Modified Algorithm 2 in Garrett Paluck's MSc thesis (quintic algorithm) 
# Input: A in Zp[x,y], 
#        F0 is an array of r polynomials in Zp[x], s.t.
#          A(y=alpha) = Lambda*prod(F0[rr],rr=1..r) mod p, for some constant Lambda, 
#        X=[x,y], variable names, 
#        alpha is in Zp, an evaluation point for y, 
#        p is a prime.
# Output: F, an Array of r polynomials in Zp[x,y] s.t. 
#         A = Lambda*product(F[rr]) mod p and F[rr](y=alpha) = F0[rho] for rr=1..r.
#         Or FAIL;
# Steps in BHL_Maple correspond to Algorithm 14 in Tian Chen's PhD thesis (2024)
BHL_Maple := proc( A::polynom, # given bivariate polynomial 
                   F0::Array(polynom), # univariate factors to be lifted
                   X::list,  # variables x,y
                   alpha::integer, # evaluation point for y=X[2]
                   p::prime, # a prime
                   f::Array(polynom) ) # Answer Array of bivariate polynomials
    local r::'integer':=numelems(F0), # number of factors
          x::'name':=X[1], # first variable
          y::'name':=X[2], # second variable
          ga::'polynom':=lcoeff(A,X[1]), # Step 1: leading coeff of A in x (polynomial in y)
          gaa::'integer', # lcoeff of a in x evaluated at y=alpha mod p
          a::'polynom', # a = ga^(r-1)*A mod p 
          f0::'list(polynom)', # updated initial factors according to the lcoeff
          df0::'list(integer)', # degrees of factors in f0
          dy::'integer', # degree of a in y 
          Mmdp::'Array(polynom)', # Array of products, as input of MDP_Maple 
          MM::'polynom', # prod(f0[rr],rr=1..r)
          M::'Array(polynom)', # MM/f0[rr], indexed rr=1..r
          rr::'integer', # loop variable
          gam::'Array(integer)', # coeffs of (y-alpha)^i in ga
          dga::'integer', # degree of ga in y
          ac::'Array(polynom)', # coeff(a,(y-alpha)^k), k=0..dy 
          sigma::'Array(polynom)', # coefficients for f[rr] 
          i::'integer', # loop variable
          k::'integer', # loop variable
          fprod::'polynom', # prod(f[rr],rr=1..r)
          Dk::'polynom', # variable for step 13
          deltak::'polynom', # variable for step 14
          ck::'polynom', # variable for step 15
          Tf::'Array(polynom)', # array of leading coefficients for step 9
          dfSum::'integer', # sum of degrees of factors in y
          fbar::'Array(polynom)', # answer from MDP_Maple at step 18
          LCFe::'integer', # scaling factor lc_eval in step 28
          eta::'integer'; # scaling factor eta in step 28
    if r <> numelems(f) then 
        error "size of the answer Array incorrect"; 
    end if;
    
    # The case of one variable: 
    if numelems(indets(A)) = 1 then # A is a polynomial in x only 
        for rr to r do 
            f[rr] := F0[rr];
        end do; 
        return; 
    end if;

    # The case of one factor: 
    if r = 1 then  
        f[1] := Primpart(A,x) mod p;
        LCFe := Eval(lcoeff(f[1],x),y=alpha) mod p;
        eta := lcoeff(F0[1],x)/LCFe mod p;
        f[1] := eta*f[1] mod p; 
        return;
    end if;
    
    # The general case of r factors (r>1): 
    a := Expand(ga^(r-1)*A) mod p; # Step 2
    gaa := Eval(ga,y=alpha) mod p;
    f0 := [seq(`mod`(gaa*F0[rr]/lcoeff(F0[rr],x),p),rr=1..r)]; # Step 3: updated initial factors
    dy := degree(a,y); 
    df0 := map(degree,f0,x); # Step 4
    
    (Mmdp,M,Tf,fbar) := Array(1..r),Array(1..r),Array(1..r),Array(1..r);
    # Mmdp is an Array as an input for MDP_Maple s.t.
    # Mmdp[r]=1, Mmdp[r-1]=f0[r],...,Mmdp[1]=f0[r]*f0[r-1]...f0[2].
    Mmdp[r] := 1;
    for i from r-1 by -1 to 1 do 
        Mmdp[i] := Expand(Mmdp[i+1]*f0[i+1]) mod p;
    end do;

    MM := Expand(Mmdp[1]*f0[1]) mod p; # Step 5: MM=prod(f0[rr],rr=1..r)
    for rr to r do 
        f[rr] := f0[rr]; # Initialize the factors 
    	M[rr] := Quo(MM,f0[rr],x) mod p; # Step 6: M[rr]=MM/f0[rr], for rr=1..r.
    end do; 
    
    gam := TaylorShift1(ga,y,alpha,p); # Step 8: gamma[k]=coeff(ga,(y-alpha)^k),k=0..dy.
    dga := degree(ga,y);
    ac := TaylorShift1(a,y,alpha,p); # Step 12: ac[k]=coeff(a,(y-alpha)^k),k=0..dy.
    sigma := Array(1..r,0..dy); # coeffs in f[rr] of (y-alpha)^k, k=0..dy
    
    # Step 11-23: the main loop
    for k to dy do  
        fprod := mul(f[rr],rr=1..r) mod p; # Quintic algorithm
        Dk := TaylorShift1(fprod,y,alpha,p,k); # Step 13: Dk=coeff(fprod,(y-alpha)^k) 
        if k <= dga then # Step 14 
            for rr to r do 
                Tf[rr] := gam[k]*x^df0[rr];
                deltak := add(Tf[i]*M[i],i=1..r); 
            end do;
        
        else 
            deltak := 0;
        end if; 
        ck := Expand(ac[k]-Dk-deltak) mod p; # Step 15

        # Step 16: 
        dfSum := add( degree(f[rr],y), rr=1..r ); 
        if dfSum = dy and ck <> 0 then 
            userinfo(4, 'BBfactor', `BHL_Maple: sum of degrees of the factors correct but error term is not zero`); 
            return FAIL; 
        end if;

        # Step 17-23:
        if ck <> 0 then 
            MDP_Maple(f0,ck,x,p,Mmdp,fbar); # Step 18: Solve MDP, answer onto fbar
            for rr to r do # step 19-21
            	if k <= dga then 
                    sigma[rr,k] := fbar[rr] + Tf[rr]; # Step 20 
        	
                else 
                    sigma[rr,k] := fbar[rr];
        	end if;
        	f[rr] := Expand(f[rr] + sigma[rr,k]*(y-alpha)^k) mod p; # Step 20 
            end do;
        end if;  
    end do;

    # Step 24:
    dfSum := add( degree(f[rr],y), rr=1..r ); 
    if dfSum <> dy then  
        userinfo(4, 'BBfactor', `BHL_Maple: sum of degrees of the factors incorrect`); 
        return FAIL; 
    end if;

    # Step 25-34:
    if ck <> 0 and Expand(a-mul(f[rr],rr=1..r)) mod p <> 0 then
        userinfo(4, 'BBfactor', `BHL_Maple: product of lifted factors does not equal to the input polynomial`); 
        return FAIL; # Step 33
    
    else 
        for rr to r do 
    	    f[rr] := Primpart(f[rr], x) mod p; # Step 27
            LCFe := Eval(lcoeff(f[rr],x),y=alpha) mod p; # Step 28 
            eta := lcoeff(F0[rr],x)/LCFe mod p; 
            f[rr] := eta*f[rr] mod p; # Step 29
        end do;
        return; # Step 31
    end if;

end:

