local CMBBSHL;
# The procedure CMBBSHL performs Hensel lifting using the Chen-Monagan sparse Hensel lifting algorithm (black box input polynomial) (Chen and Monagan, ISSAC 2023).
# Input: A modular black box B representing the polynomial a(x1,...,xn) in Z[x1,...,xn],
#        initial factors in one variable (the main variable) mod p, 
#        evaluation point alpha=[alpha_2,...,alpha_n], for the variables x2,...,xn,
#        pre-computed degrees of a in x_j, j=1..n,
#        a large prime p.
# Ouput: The factors of a(x1,...,xn) in n variables mod p, w.h.p. correct, or FAIL or false.
#        FAIL indicates an unlucky evaluation, false implies to use a large integer case instead.

CMBBSHL := proc( B::procedure, # modular black box for a(x1,...,xn)
                 f1::Array(polynom),  # Array of initial univariate polynomials as input for Hensel lifting
                 NN::posint, # variable counter for recursive calls
                 X::list(name),    # list of variables x1,...,xn
                 alpha::Array, # evaluation point
                 deg::list,   # partial degrees of a(x1,...,xn)
                 p::prime,  # main prime
                 singlepow::nonnegint, # indicator if a power (>0) of the first variable exists
                 MapleCode::table,  # use C or Maple code for the 4 subroutines in CMBBSHL_stepj
                 $ )::{Array(polynom),symbol};
    global interp2var,BHL;
    local LI::'truefalse':=`if`(p>2^61,true,false), # Boolean for large integer case, decided if p>2^61.
          r::'integer':=numelems(f1), # number of factors for Hensel lifting
          rr::'integer', # loop variable
          Sdf1::'integer', # sum of degrees of x1 in f1
          du::'integer', # max of the degrees in x1 in all factors
          d1s::'integer':=deg[1]-singlepow, # deg(a,x1) - singlepow
          dmax::'integer', # max(d1s,deg(a,x2))
          xx::'Array(integer)', # Array of evaluation points
          II::'Array(integer)', # Array for storing Newton inverses
          MM::'Array(integer)', # 2d Array to store evaluations of a(x1,x2,alpha_3,...,alpha_n) mod p
          dy::'integer', # degree of square-free part of a(x1,x2,alpha_3,...,alpha_n) mod p in x2
          MMsqf::'Array(integer)', # 2d Array for evaluations of sqf(a(x1,x2,alpha_3,...,alpha_n)) mod p
          a2::'polynom', # interpolated polynomial a2 := a(x1,x2,alpha_3,...,alpha_n) mod p
          Asf::'polynom', # square-free part of a(x1,x2,alpha_3,...,alpha_n) mod p
          f0A::'Array(integer)', # 2d Array storing the coefficients of f1 before bivariate Hensel lifting
          fA::'Array(integer)', # 2d Array storing answers after bivariate Hensel lifting (BHL)
          sizeW::'integer', # size of temp storage Array for BHL C code 
          Wtemp::'Array(integer)', # temp storage for BHL C code
          DDX::'Array(integer)', # Input Array for BHL C code
          DDY::'Array(integer)', # Input Array for BHL C code
          BHL_ans::{'integer','symbol'}, # Output of BHL code  
          f::'Array(polynom)', # answer as an Array of polynomials after BHL  
          ii::'integer', # loop variable
          jj::'integer', # loop variable
          fnm1::{'Array(polynom)','symbol'}, # answer of recursive calls as a list of polynomial or FAIL or false
          DataTypeInt8:='datatype'='integer'[8],
          st:=time(), # timer
          et; # timer
    if NN = 2 then # When only two variables present, perform a bivariate Hensel lift (BHL) 
    
        if MapleCode[interp2var] then # Use Maple code to get a2 := a(x1,x2,alpha_3,...,alpha_n) mod p 
            a2 := Interp2var_Maple( B, X, alpha, [1,2], deg, p, singlepow, LI );    

        else # Use C code to get a2
            dmax := max(d1s,deg[2]); # maximum of deg(a,x1)-singlepow and deg(a,x2)
            xx := Array(0..dmax, i->i+1, DataTypeInt8); # Points for interpolation. Avoided 0 for the case singlepow > 0 
            II := Array(0..dmax, DataTypeInt8); # Answer for NewtonInv (C code)
            NewtonInv( xx, dmax+1, II, p ); # For Newton interpolation, compute the inverses of proc(xx[j]-xx[i],i=1..j-1) for j=1..dmax. 
	    MM := Array(0..(d1s+1)*(deg[2]+1)-1,DataTypeInt8);  
	    GetEvaluations( B, alpha, [1,2], [d1s,deg[2]], xx, MM, p, singlepow ); # C code for evaluation, MM[i][j]=a2(i,j) mod p
            Interp2var( 1,2,d1s,deg[2],dmax,xx,II,MM,p ); # C code to interpolate a2, MM[i][j]=coeff of x1^(i+singlepow)*x2^j in a2
            a2 := Expand(add(add(MM[(d1s+1)*jj+ii]*X[1]^ii,ii=0..d1s)*X[2]^jj,jj=0..deg[2])) mod p; # Put a2 into a polynomial
        end if;
            
        if degree(a2,X[1])<>d1s or degree(a2,X[2])<>deg[2] then 
            userinfo(4, 'BBfactor', `Initial BHL: unlucky evaluation at interpolating`, a2);
            return FAIL;  
	end if;

        # Compute the square-free part of a2 by a gcd and a division
        Gcd(a2,diff(a2,X[1]),'Asf') mod p;
        Asf := Asf/lcoeff(lcoeff(Asf,X[1]),X[2]) mod p;     
        
        Sdf1 := add(degree(f1[rr],X[1]),rr=1..r);  # sum of degrees of x1 in f1 
        if degree(Asf,X[1])<>Sdf1 then 
            userinfo(4, 'BBfactor', `unlucky evaluation at computing square-free part of`, a2, `:wrong degree in`, X[1]);
            return FAIL;  
	end if;
         
        dy := degree(Asf,X[2]); # degree of Asf in x2 
        f := Array(1..r); # answer array of bivariate polynomials for BHL 
        # Perform bivariate Hensel lifting (BHL)
        if MapleCode[BHL] or r = 1 or numelems(indets(Asf)) = 1 then 
            # Use Maple code for BHL if only one factor or one variable is present
            BHL_ans := BHL_Maple( Asf, f1, [X[1],X[2]], alpha[2], p, f ); # list of bivariate factors 
            et := time()-st; 
            userinfo(5, 'BBfactor', `To recover`,X[2],`total time for initial BHL = `,et);
            
            if BHL_ans = FAIL then 
                userinfo(4, 'BBfactor', `Bivariate Hensel lifting (Maple program) failed, likely due to a Non-Hilbertian point`); 
                return FAIL; 
            end if;
            return f;

 	else # Use C code otherwise 
            MMsqf := Array(0..(Sdf1+1)*(dy+1)-1,DataTypeInt8);
            BivarPoly2matrix( Asf, [X[1],X[2]], Sdf1, MMsqf ); # MMsqf[i][j]=coeff of X[1]^i*X[2]^j in Asf
            du := max( map( degree, f1, X[1] ) ); 
            f0A := Array(1..r,0..du,DataTypeInt8,'order'='C_order'); # Input array (coeffs of f1(X[1])) before BHL
            Initialf2matrix( f1, X[1], f0A );
            fA := Array(1..r,0..2*(Sdf1+1)*(dy+1)-1,DataTypeInt8,'order'='C_order'); # Answer array after BHL
            sizeW := allocateSpace(r,Sdf1,dy,du);
            Wtemp := Array(1..sizeW,DataTypeInt8,'order'='C_order'); # Wtemp is temp work space for procedure BHL_C
            # DDX and DDY are degrees of the factors 
            DDX := Array(1..r,0..dy,DataTypeInt8,'order'='C_order');
            DDY := Array(1..r,DataTypeInt8,'order'='C_order');
            BHL_ans := BHL_C( MMsqf, Sdf1, dy, DDX, DDY, f0A, r, du, fA, alpha[2], Wtemp, p ); # BHL using C code
            # fA is the answer of BHL_C, the coefficients of the bivairate factors
            
            if BHL_ans = -1 then 
                userinfo(4, 'BBfactor', `Bivariate Hensel lifting (C program) failed, likely due to a Non-Hilbertian point`); 
                return FAIL; 
            end if;

            # Answer of BHL as an Array of polynomials
            for rr to r do 
                f[rr] := Expand(add(add(fA[rr][(Sdf1+1)*jj+ii]*X[1]^ii,ii=0..DDX[rr,jj])*X[2]^jj,jj=0..DDY[rr])) mod p; 
            end do;
            et := time()-st; 
            userinfo(5, 'BBfactor',`To recover`,X[2],`total time for initial BHL = `,et);
            return f;
        
        end if; # Maple or C code
    
    end if; # NN=2
    
    # Recursively call CMBBSHL for NN-1 variables
    fnm1 := CMBBSHL( B, f1, NN-1, X, alpha, deg, p, singlepow, MapleCode );
    
    if fnm1 = FAIL or fnm1 = false then 
        return fnm1; 
    end if;
    # Lift the factors from NN-1 to NN variables 
    return CMBBSHL_stepj( B, fnm1, NN, X, alpha, deg, p, singlepow, MapleCode );
end proc;
