# The procedure CMBBSHL_stepj performs the jth (j>2) Hensel lifting step in the procedure CMBBSHL.
# This is Algorithm 13 in PhD thesis of Tian Chen (2024)
# Input: The black box B representing the polynomial a in Z[x1,...,xn],
#        the factors of a in j-1 variables (x1,...,x_{j-1}) (j>2) mod p,
#        evalution point alpha_j in Zp (passed in as alpha=[alpha_2,...,alpha_n]),
#        degree of a in xj (passed in as a list of partial degrees of a), 
#        a prime p.
# Output: The factors of a in j (j>2) variables mod p, w.h.p. correct, or FAIL or false.
#        FAIL indicates an unlucky evaluation, false implies to switch to large integer case in the BBfactor proc instead. 
local CMBBSHL_stepj;
CMBBSHL_stepj := proc( B::procedure, # A modular black box B for a(x1,...,xn)
                       fjm1::{Array(polynom),symbol}, # Array of factors of a in j-1 variables mod p
                       j::integer,    # Variable index, the jth variable (2<j<=n) for Hensel lifting
                       X::list(name), # List of variables x1,...,xn
                       alpha::Array,  # evaluation point alpha=[alpha_2,...,alpha_n]
                       deg::list,     # deg(a,xj) for j=1..n.
                       p::prime,      # The main prime p
                       singlepow::nonnegint, # If a power of x1 exists as a factor, singlepow>0 is the exponent; else = 0
                       MapleCode::table, # indicators whether to use Maple or C code in the 4 subroutines
                       $ )::{Array(polynom),symbol}; 
    global interp2var, VS, BHL, evalfac;
    local LI::'truefalse':=`if`(p>2^61,true,false), # Boolean for large integer case, decided if p>2^61.
          r::'integer':=numelems(fjm1), # number of factors
          N::'posint':=numelems(X), # Number of variables in X=[x1,...,xn] (N=n) 
          rr::'integer',  # loop index
          Sdf1::'integer':=add(degree(fjm1[rr],X[1]),rr=1..r), # Sum of the degrees in X[1] in fjm1  
          ntf::'list(integer)':=map(nops,fjm1),  # number of terms in each factor
          F::'Array',  # Address of each polynomial fjm1, indexed from 1 to r
          sigma::'Array(list)', # Coefficients of fjm1 in x1, stored as a list in descending degrees of x1 for each factor, indexed rr=1..r 
          monfx1::'polynom', # monomials in sigma[rr] 
          degfx1::'Array(list)', # degrees of x1 (corresponding to sigma), stored as a list for each factor, indexed rr=1..r 
          df1::'Array(integer)', # maximum degree of x1 of each factor, indexed rr=1..r
          M::'Array(list)', # monomials (in x2,...,xn) in each coefficient of fjm1 in x1, stored as a list of list for each factor, indexed rr=1..r 
          s::'Array(list)', # number of monomials (in x2,...,xn) in each coefficient of fjm1 in x1, stored as list for each factor, indexed rr=1..r 
          S::'Array(list)', # monomial evalutations of M, stored as a list of list, indexed rr=1..r
          k::'integer', # loop index
          test::'truefalse', # indicator for looping to choose an evaluation point 
          betaA::'Array(integer)', # evaluation point betaA=[beta_1...,beta_j] in an Array
          betaAm::'Array(integer)', # evaluation point when the number of variables in fjm1[rr] < j-1
          Y::'list', # Y=[X[i]=beta[i],i=2..j-1]
          ss::'integer', # maximum number of monomials in sigma[rr]_i among all factors, sigma[rr]_i is the coeff of fjm1[rr] in x^i
          univar::'Array(truefalse)', # Indicators if a factor is univariate
          FA::'Array(Array)', # For eval fjm1: stores the coefficients of X[1] for each factor fjm1, evaluated at X[2]=beta_2,...,X[j-1]=beta_{j-1}
          TA::'Array(Array)', # For eval fjm1: stores the number of monomials in the coefficients of X[1] for each factor fjm1
          CA::'Array(Array)', # For eval fjm1: Results of monomial evaluations times the corresponding integer coefficients
          MA::'Array(Array)', # For eval fjm1: Monomial evaluations of M[rr] at Y = [X[2]=beta_2,...,X[j-1]=beta_{j-1}]
          vars::'set(name)', # list of variables in fjm1[rr]
          nvars::'integer', # number of variables in fjm1[rr]
          CAlist::'list', # For computing CA
          MAlist::'list', # For computing MA
          b::'Array(integer)', # evaluation points beta_k:=[beta_1^k,...,beta_j^k,alpha_{j+1},...,alpha_n], k=1..ss.
          dj::'integer':=deg[j], # deg(a,xj)
          d1s::'integer':=deg[1]-singlepow, # deg[1] - singlepow
          dmax::'integer',  # max(d1s,dj)
          xx::'Array(integer)', # points for interpolation
          II::'Array(integer)', # Computed inverses for Newton interpolation
          MM::'Array(integer)', # answer array for interpolating a2_k:=a(x1,beta_2^k,...,beta_{j-1}^k,xj,alpha_{j+1},...alpha_n) mod p
          MMsqf::'Array(integer)', # array to store the coefficients of square-free part of a2_k(x1,xj)
          a2::'polynom', # stores the interpolated polynomial a2_k, for k=1..ss
          Asf::'polynom', # square-free part of a2_k
          dy::'integer', # deg(Asf,xj)
          du::'integer', # max(df1[rr]), maximum degree in x1 among the factors
          fA::'Array(integer)', # answer array for BHL C code
          f0A::'Array(integer)', # input array for BHL C code, fjm1 evaluated at beta_k
          Wtemp::'Array(integer)', # Temp work space for BHL C code
	  sizeW::'integer', # size of Wtemp
          DDX::'Array(integer)', # Input array for BHL C code
          DDY::'Array(integer)', # Input array for BHL C code
          BHL_ans::'integer', # Output from BHL code
          i::'integer', # loop variable
          ii::'integer', # loop variable
          jj::'integer', # loop variable
          ll::'integer', # loop variable (lower case double L's)
          f0::'Array(polynom)', # Evaluated fjm1 at Yk = [X[2]=beta_2^k,...,X[j-1]=beta_{j-1}^k]
          f::'Array(polynom)', # Result of BHL, an Array of bivariate polynomials
          nf::'Array(list)', # Array of lists which consists number of terms of the bivariate factors
          Mon::'list(polynom)', # list of monomials (bivariate)
          Cof::'list(integer)', # list of integer coefficients corresponding to Mon
          mon::'polynom', # individual monomial in Mon
          cof::'integer', # individual coefficient in Cof
          Tf::'Array(table)', # tables storing monomials with corresponding coefficients of bivariate factors
          TCOF::'Array(integer)', # Array of coefficients for a particular bivariate monomial computed by BHL
          SA::'Array(integer)', # For VSolve in C: monomial evaluations
          trr::'integer', # For VSolve: maximum number of terms among all ss number of f[rr]'s  
          dx1::'integer', # For VSolve: degree of the bivariate monomial in x1
          id::'integer', # index for dx1 
          sizeVS::'integer', # size of Vandermonde system
          Mtemp::'Array(integer)', # For VSolve in C: temp space
          Qtemp::'Array(integer)',  # For VSolve C in C: temp space
          C::'Array(integer)', # Result of Vandermond solves
          termsf::'table', # Expanded terms for coefficients in x1^ii*x^jj, for some ii,jj
          fN::'Array(polynom)', # final factors in j variables
          fY::'Array(polynom)', # for check answer, evaluated factors
          a::'polynom', # for check answer, an interpolated univariate polynomial
          DataType, # for Arrays    
          DataInt8:='datatype'='integer'[8],
          DataInt4:='datatype'='integer'[4];

    local st0:=time(),tt2:=0,tt3:=0,bbtime:=0,interptime:=0,vstime:=0,tbb,tinterp,tvs,st,et;
    
    # The step numbers in the comments refer to Algorithm 13 in PhD Thesis of Tian Chen (2024)     
    
    # Step 1: Initial computations (fjm1 is the input list of factors in j-1 variables)
    (F,sigma,degfx1,df1,M,s,S) := Array(1..r),Array(1..r),Array(1..r),Array(1..r),Array(1..r),Array(1..r),Array(1..r); 
    for rr to r do  
        F[rr] := addressof(fjm1[rr])-4*(2^63-1); # To determine the DAG type, Address of each factor fjm1
        sigma[rr] := [coeffs(fjm1[rr], X[1], 'monfx1')]; # Coefficients of fjm1 in x1, stored as a list in descending degrees of x1 for each factor (1 to r) 
        degfx1[rr] := map(degree, [monfx1], X[1]); # degrees of x1 (corresponding to sigma), stored as a list for each factor (1 to r) 
	df1[rr] := max(degfx1[rr]);  # maximum degree of x1 in each factor
        M[rr] := map(support, sigma[rr], X[2..j-1]); # monomials (in x2,...,xn) in sigma[rr], stored as a list of list for each factor 
        s[rr] := map(nops, M[rr]); # number of monomials for each element in M[rr], stored as list for each factor
    end do;

    # Step 2: Choose an evaluation point betaA=[beta_1,...,beta_j] and check for distinct evaluations
    DataType := `if`(not LI, DataInt8, NULL); 
    betaA := Array(1..j,DataType);
    for k to 20 do 
        test := true;  
        for i to j do 
            betaA[i] := rand(p)();
        end do;
        # Step 3
        Y := [seq(X[i]=betaA[i], i=2..j-1)]; # Y = [X[2]=beta_2,...,X[j-1]=beta_{j-1}]
        for rr to r do 
            S[rr] := map( Eval, M[rr], Y ) mod p; # Step 3: Monomial evaluations of M[rr] at Y
            for i to numelems(sigma[rr]) do 
	        if numelems({op(S[rr][i])}) < s[rr][i] then # Step 4: Check for distinct evaluations 
                    test := false; 
                    next k; 
                end if; 
	    end do;
        end do;
    until test = true; 
    if k > 20 then # Step 4 
        userinfo(4, 'BBfactor', `CMBBSHL step: `,j,`monomial evaluations are not unique, prime is likely too small`); 
        return false; 
    end if;
    
    # Step 5: Let ss be the maximum number of monomials in sigma[rr]_i among all factors, sigma[rr]_i is the coeff of fjm1[rr] in x1^i
    ss := max([seq(s[rr],rr=1..r)]); 
    userinfo(5, 'BBfactor', `CMBBSHL step: `,j,`s = `,ss);
    
    univar := Array(1..r,i->false); # indicator if a factor is univariate

    # Set up for evaluations of fjm1, if using C code
    if MapleCode[evalfac] = false then
        # E.g. p = 19, alpha = [3,5] (y=3,z=5) 
        # f[rr] = 3x^2y^2 + 5z^3y + 2x^2 + 4yz  = (5yz^3 + 4yz)x^0 + 0*x^1 + (3y^2+2)x^2 
        #  CA[rr] = [5,4,3,2] (=[(5,4),(),(3,2)]) # coefficients of each monomial in x^i, i=0..2.
        #  MA[rr] = [14,15,9,1] (=[yz^3,yz),(),(y^2,1)]_{y=3,z=5}) # monomial evaluations corresponding to CA
        #  After evalnext64s (in nextimage): CA[rr] = [13,3,8,2] (CA[i]:=CA[i]*MA[i] mod p, i=1..4)	
        #  TA[rr] = [2,0,2] # Number of terms for each degree of x
        #  FA[rr] = [16,0,10] # Result after evaluation (16=CA[1]+CA[2]=13+3,10=CA[3]+CA[4]=8+2)
        
        (MA,CA,TA,FA) := Array(1..r),Array(1..r),Array(1..r),Array(1..r); 
        for rr to r do 
	    MA[rr] := Array(1..ntf[rr],DataInt8); # Monomial evaluations of M[rr] at Y = [X[2]=beta_2,...,X[j-1]=beta_{j-1}]
            CA[rr] := Array(1..ntf[rr],DataInt8); # Results of monomial evaluations times the corresponding integer coefficients
	    TA[rr] := Array(0..df1[rr],DataInt4); # Number of monomials present in sigma[rr]_i stored for each degree of X[1], in ascending order 
	    FA[rr] := Array(0..df1[rr],DataInt8); # Results of the evaluation of fjm1[rr], stored for each degree of X[1], in ascending order
            vars := indets(fjm1[rr]); # variables in fjm1[rr] 
            nvars := nops(vars); # number of variables in fjm1[rr]
 
            if getmapleID(F[rr])<>17 then # fjm1[rr] is not POLY DAG 
                if nvars = 1 then # fjm1[rr] is univariate 
                    univar[rr] := true; 
                    k := 1;
 		    for i in degfx1[rr] do 
                        TA[rr][i] := 1; 
                        FA[rr][i] := sigma[rr][k]; # The result of evaluation, copied from sigma[rr] 
                        k++: 
                    end do; 
                    for i to ntf[rr] do 
                        MA[rr][i] := 1; 
                    end do;
                
                    if df1[rr] = 1 then # fjm1[rr] is univariate and is linear
                        for i to ntf[rr] do 
                            CA[rr][i] := sigma[rr][i]; 
                        end do; 

                    else # fjm1[rr] is univariate but non-linear, i.e. df1[rr] > 1 
                        # sigma[rr] stores the coeffs in descending order of degrees in X[1], for CA[rr] switch to ascending order 
                        for i to ntf[rr] do 
                            CA[rr][i] := sigma[rr][ntf[rr]-i+1];
                        end do;
                    end if;

	        else # fjm1[rr] is linear but not univariate 
		    k := 1; 
                    for i in degfx1[rr] do 
                        TA[rr][i] := s[rr][k]; # number of monomials in sigma[rr]_i 
                        FA[rr][i] := Eval(sigma[rr][k],Y) mod p; # Result of evaluation  
		        k++; 
                    end do;
                    CAlist := map(coeffs,sigma[rr],X[2..j-1]); 
		    MAlist := map(op,S[rr]);
                    for i to ntf[rr] do 
	                CA[rr][i] := CAlist[i]; 
		        MA[rr][i] := MAlist[i]; 
                    od;   
	        end if;

	    else # fjm1[rr] is POLY DAG
                if nvars < j-1 then 
	            betaAm := Array(1..nvars,DataInt8); 
                    k := 1;
                    for i to j-1 do 
	                if member(X[i],vars) then 
	                    betaAm[k] := betaA[i]; 
                            k++; 
                        end if; 
		    end do;
                    # getsupport calls external C program to compute monomial evaluations, result onto MA[rr]
                    getsupport(F[rr],ntf[rr],betaAm,nvars,1,df1[rr],TA[rr],CA[rr],MA[rr],p); 
                else
                    getsupport(F[rr],ntf[rr],betaA[1..j-1],j-1,1,df1[rr],TA[rr],CA[rr],MA[rr],p);
                end if;
            end if; 
        end do;
    end if;
    
    # b stores the evaluation points [beta_1^k,...,beta_j^k,alpha_{j+1},...,alpha_n] for the main loop k=1..ss. 	 
    b := Array(1..N,i->`if`(i<=j,betaA[i],alpha[i]),DataType);
 
    # Compute a square free image of a2:=aj(x1,beta_2,...,beta_{j-1},xj) before the main loop for 
    # 1) deciding its degrees in x1 and xj, 
    # 2) allocating memories for BHL C code.  
    if MapleCode[interp2var] then # Use Maple code to get a2 (=a2_k where k>=1)   
         a2 := Interp2var_Maple( B, X, b, [1,j], deg, p, singlepow, LI ); 
   
    else # Use C code to get a2 
        dmax := max(d1s,dj);  # max(d1s,dj)
        xx := Array(0..dmax, i->i+1, DataInt8); # Points for interpolation. Avoid 0 for the case singlepow > 0 
        II := Array(0..dmax, DataInt8); # Answer for NewtonInv (C code) 
        NewtonInv( xx, dmax+1, II, p ); # For Newton interpolation, compute inverses of prod(xx[j]-xx[i],i=1..j-1), j=1..dmax. 
        MM := Array(0..(d1s+1)*(dj+1)-1, DataInt8);
        GetEvaluations( B, b, [1,j], [d1s,dj], xx, MM, p, singlepow ); # Use C code for evaluations, MM[i][j] = a2(i,j) mod p
        Interp2var( 1, j, d1s, dj, dmax, xx, II, MM, p ); # Use C code to interpolate a2, MM[i][j]=coeff of x1^(i+singlepow)*xj^j of a2
        a2 := Expand(add(add(MM[(d1s+1)*jj+ii]*X[1]^ii,ii=0..d1s)*X[j]^jj,jj=0..dj)) mod p; # Put a2 into a polynomial 
        clearArray(MM,0,(d1s+1)*(dj+1)-1); 
    end if;

    if degree(a2,X[1])<>d1s or degree(a2,X[j])<>dj then 
        userinfo(4, 'BBfactor', `unlucky evaluation at interpolating`, a2);
        return FAIL; 
    end if;
             
    Gcd(a2,diff(a2,X[1]),'Asf') mod p; 
    Asf := Asf/lcoeff(lcoeff(Asf,X[1]),X[j]) mod p; 
        
    if degree(Asf,X[1])<>Sdf1 then 
        userinfo(4, 'BBfactor', `unlucky evaluation at computing square-free part of`, a2, `:wrong degree in`, X[1]);
        return FAIL;  
    end if;     
    dy := degree(Asf,X[j]); # deg(Asf,xj), Asf is the square-free part of a2
 
    # Memory allocation for BHL C code
    if MapleCode[BHL] = false and r > 1 and numelems(indets(Asf)) > 1 then 
        MMsqf := Array(0..(Sdf1+1)*(dy+1)-1, DataInt8); # MMsqf[i][k] stores the coefficient of X[1]^i*X[j]^k in Asf
        du := max(seq(df1[rr],rr=1..r)); 
        f0A := Array(1..r,0..du, DataInt8,'order'='C_order'); # Input array (coeffs of f0(X[1])) before BHL
        fA := Array(1..r,0..2*(Sdf1+1)*(dy+1)-1,DataInt8,'order'='C_order'); # Answer array after BHL
        sizeW := allocateSpace( r, Sdf1, dy, du );
        Wtemp := Array(1..sizeW, DataInt8 ); # Wtemp is temp work space for BHL_C
        DDX := Array(1..r,0..dy, DataInt8,'order'='C_order'); # DDX and DDY store the degrees of the factors 
        DDY := Array(1..r, DataInt8);
    end if;

    # Initialize Tf, Array of tables to store the coefficients and monomials of the bivariate factors after BHL 
    Tf := Array(1..r,i->table()); 
    nf := Array(1..ss);
    (f0,f) := Array(1..r),Array(1..r); # f0 and f are arrays of univariate and bivariate polynoimals respectively. 
 
    # b stores the evaluation point [beta_1^k,...,beta_j^k,alpha_{j+1},...,alpha_N]  
    b := Array(1..N,i->`if`(i<=j,1,alpha[i]),DataType);
    
    # Step 6-18: Main loop for k from 1 to ss which contains 3 subroutines: 
    # 1) Evaluating fjm1, 2) Interpolating the square-free part of a2, 3) Bivaraite Hensel lifting  
    for k to ss do
        # Step 7: Evaluation point b = [beta_1^k,...,beta_j^k]
        for i to j do 
            b[i] := betaA[i]*b[i] mod p;
        end do;
        
        # Step 8-13: Compute a square free part of the image a2_k:=a(x1,Yk,xj,alpha_{j+1},...,alpha_n) mod p, Yk=[x_2=beta_2^k,...,x_j=beta_j^k]
        # Step 8: Compute a bivariate image a2_k:=a(x1,Yk,xj)/x1^singlepow using dense interpolation
        if MapleCode[interp2var] then # Use Maple code to get a2 (=a2_k where k>=1)   
            tinterp := time();   
            a2 := Interp2var_Maple( B, X, b, [1,j], deg, p, singlepow, LI ); 
            interptime+=time()-tinterp;
   
        else # Use C code to get a2 
            tbb := time();     
            GetEvaluations( B, b, [1,j], [d1s,dj], xx, MM, p, singlepow );  # MM[i][j]=a2_k(i,j) mod p
            bbtime+=time()-tbb;

            tinterp := time();  
            Interp2var( 1, j, d1s, dj, dmax, xx, II, MM, p ); # MM[i][j]=coeff of x1^(i+singlepow)*xj^j in a2_k
            interptime+=time()-tinterp;
                
            a2 := Expand(add(add(MM[(d1s+1)*jj+ii]*X[1]^ii,ii=0..d1s)*X[j]^jj,jj=0..dj)) mod p; # Put a2_k into a polynomial   
	    clearArray( MM,0,(d1s+1)*(dj+1)-1 );
        end if;

        # Step 9: check degrees of a2 
        if degree(a2,X[1])<>d1s or degree(a2,X[j])<>dj then 
            userinfo(4, 'BBfactor', `unlucky evaluation at interpolating`, a2);
            return FAIL; 
        end if;
 	    
        # Step 10-13: Compute the square-free part of a2_k (=Asf) by a gcd and a division
        Gcd(a2,diff(a2,X[1]),'Asf') mod p; 
        Asf := Asf/lcoeff(lcoeff(Asf,X[1]),X[j]) mod p; 
        
        # Step 11: Check degrees of Asf  
        if degree(Asf,X[1])<>Sdf1 or degree(Asf,X[j])<>dy then 
            userinfo(4, 'BBfactor', `unlucky evaluation at computing square-free part of`, a2);
            return FAIL;  	
        end if;			
         
        # Step 14: Evaluate fjm1 at Yk = [x[2]=b[2]^k,...,x[j-1]=b[j-1]^k]
        st := time();
        if MapleCode[evalfac] then # Use Maple code to evaluate fjm1 at Yk
            for rr to r do  
                f0[rr] := Eval(fjm1[rr],[seq(X[ii]=b[ii],ii=2..j-1)]) mod p;
            end do;
 
            if MapleCode[BHL] = false and r > 1 and numelems(indets(Asf)) > 1 then 
                Initialf2matrix(f0,X[1],f0A); # After evaluation, put f0 into f0A for BHL C code
            end if;
        
        else # Use C code to evaluate fjm1 at Yk
            # If f0[rr] is univariate, no need for evaluation; else call C program nextimage, result is on FA[rr]
            for rr to r do 
                f0[rr] := `if`(not univar[rr], nextimage(CA[rr],MA[rr],ntf[rr],TA[rr],df1[rr],FA[rr],X[1],p), fjm1[rr]);
            end do; 
                
            # For BHL C code
            for rr to r do 
                for ii from 0 to df1[rr] do 
                    f0A[rr,ii] := FA[rr][ii]; 
                end do;
            end do;  
        end if;
        et := time()-st; tt2 += et;

        # Step 15: check degrees for each evaluated factor 
        for rr to r do 
            if degree(f0[rr],X[1]) < df1[rr] then 
                userinfo(4, 'BBfactor', `unlucky evaluation at evaluating the factors`, f0[rr], `:wrong degree in`, X[1]);
                return FAIL; 
            end if; 
        end do;
 
        # Step 16: Check if the evaluated factors are relatively prime 
        # Check if product of f0's is square free instead? 
        for ii to r do
            for jj from ii+1 to r do
                if Gcd(f0[ii],f0[jj]) mod p <> 1 then 
                    userinfo(4, 'BBfactor', `CMBBSHL step`,j,`evaluated factors are not relatively prime`);
                    return FAIL;
                end if;
            end do;
        end do;

        # Step 17: Bivariate Hensel lifting (BHL) 
        st := time();
	if MapleCode[BHL] or r = 1 or numelems(indets(Asf)) = 1 then # Use Maple code for BHL
            BHL_ans := BHL_Maple( Asf, f0, [X[1],X[j]], alpha[j], p, f ); 
            # Result f is an Array of bivariate factors

            if BHL_ans = FAIL then 
                userinfo(4, 'BBfactor', `Bivariate Hensel lifting (Maple program) failed, likely due to a Non-Hilbertian point`);
                return FAIL;   
            end if; 
           
	else # Use C code for BHL 
            BivarPoly2matrix( Asf, [X[1],X[j]], Sdf1, MMsqf ); # Put the coefficients of Asf onto MMsqf
            BHL_ans := BHL_C( MMsqf, Sdf1, dy, DDX, DDY, f0A, r, du, fA, alpha[j], Wtemp, p ); 
            # BHL C code, result stored onto fA, coefficients of the bivariate factors.
 
            if BHL_ans = -1 then 
                userinfo(4, 'BBfactor', `Bivariate Hensel lifting (C program) failed, likely due to a Non-Hilbertian point`);
                return FAIL;   
            end if; 
            
            # Convert the result from BHL C program to a list of bivariate polynomials f = [f1,...,fr]
            for rr to r do 
                f[rr] := Expand(add(add(fA[rr][(Sdf1+1)*jj+ii]*X[1]^ii,ii=0..DDX[rr,jj])*X[j]^jj,jj=0..DDY[rr])) mod p;
            end do;
            
            clearArray2d( f0A,1,r,0,du ); 
            clearArray2d( fA,1,r,0,2*(Sdf1+1)*(dy+1)-1 );
            clearArray( MMsqf,0,(Sdf1+1)*(dy+1)-1 ); 
            clearArray( Wtemp,1,sizeW ); 
        end if;
        et := time()-st; tt3 += et;
        
        # Step 19: Store coefficients and monomials of f[rr] (bivariate polynomial) to Tf[rr] (tables)
        # E.g. f = [x1^2 + 2*x1*x3 + 8*x1 + 7, x1 + 5]  
        # nf = [4, 2], number of terms for each factor in f
        # Cof = [1,2,8,7], Mon = [x1^2,x1*x3,x1,1]
        # Tf[1]=table([x1^2=[1],x1*x3=[2],x1=[8],1=[7]]), 
        # Tf[2]=table([1=[5],x1=[1]])  
        nf[k] := map( nops, f );
        for rr to r do 
            Cof, Mon := MonCoffs( f[rr], [X[1],X[j]] ); # Output coefficients and monomials as two lists
            for i to nops(Mon) do
                mon := Mon[i]; 
                cof := Cof[i]; 
                if not assigned(Tf[rr][mon]) then
                    Tf[rr][mon] := Array(1..ss); 
                end if;
                Tf[rr][mon][k] := cof;
            end do;
        end do;
    end do;
     
    # Step 20-26: Solving Vandermonde systems to recover the factors in j variables. 
    # For each monomial x1^ii*xj^jj (for some power ii,jj) in the bivariate factors after BHL, 
    # solve a Vandermonde system to recover the coefficients corresponding to the monomials 
    # (in x2,...,x_{j-1}) in sigma[rr]_i's. 
 
    fN := Array(1..r);
    if MapleCode[VS] then # Use Maple code for Vandermonde solves 
	for rr to r do 
            trr := max( [seq(nf[k][rr],k=1..ss)] ); # maximum number of terms among ss number of f[rr]'s (bivariate images)  
            for ll to trr do 
                dx1 := degree(Tmon(Tf,r,ll,rr),X[1]); # degree of the monomial x1^ii*xj^jj in x1, for some ii, jj 
		id := ListTools:-Search(dx1,degfx1[rr]); # Step 22: index of dx1 in degfx1[rr]
		sizeVS := s[rr][id]; # number of monomials in sigma[rr] corresponding to that degree of x1 
		TCOF := Tcof(Tf,r,ll,rr,LI); # Array of coefficients of x1^ii*xj^jj computed from BHL for all s images 
   
                # Step 23: Solve the Vandermond system in Maple, VC = b, where C is the result, b is TCOF, 
                # V is a sizeVS by sizeVS shifted transposed Vandermonde matrix, shift=1.
                # shift=1: The first row of V is the monomial evaluations in S[rr][id], instead of 1's.  
		tvs:=time();
                C := VandermondeSolve_Maple(S[rr][id], TCOF, sizeVS, 1, p);  
		vstime+=time()-tvs;
                # Construct result polynomials in x2,...,x_{j-1}:
                termsf[ll] := add(C[k]*M[rr][id][k], k=1..sizeVS);  

            end do;
            # Step 25: Expand to get the final factors in x1,...,xj
            fN[rr] := Expand(add(termsf[ll]*Tmon(Tf,r,ll,rr), ll=1..trr)) mod p; 
        end do;
	
    else # Use C code for Vandermonde solves
        Qtemp := Array(0..ss,DataInt8);
        for rr to r do 
            trr := max( [seq(nf[k][rr],k=1..ss)] ); # maximum number of terms among ss number of f[rr]'s   
            for ll to trr do
                dx1 := degree(Tmon(Tf,r,ll,rr),X[1]); # degree of the monomial in x1 
                id := ListTools:-Search(dx1,degfx1[rr]); 
                sizeVS := s[rr][id]; # number of monomials in sigma[rr] corresponding to that degree of x1 
		TCOF := Tcof(Tf,r,ll,rr,LI);  # Array of coefficients computed from BHL 
                
                # SA, Mtemp are work spaces for VSolve in C
                SA := Array(S[rr][id],DataInt8); 
                Mtemp := Array(0..sizeVS,DataInt8);  
                C := Array(1..sizeVS,DataInt8); 
                
                # Vandermonde solve in C program, answer onto the array C
                tvs:=time();
                VSolve( SA, TCOF, sizeVS, C, Mtemp, Qtemp, 1, p ); 
                vstime+=time()-tvs;

                termsf[ll] := add(C[k]*M[rr][id][k], k=1..sizeVS);
            end do;
            # Step 25: Expand to get the final factors in j variables
            fN[rr] := Expand(add(termsf[ll]*Tmon(Tf,r,ll,rr), ll=1..trr)) mod p; 
        end do;
    end if;

    if j = N then	
        userinfo(5, 'BBfactor', `Timings: BB(total),Interp2var,Eval,BHL,VS`,bbtime,interptime,tt2,tt3,vstime );
    end if;

    # Step 27-30: Check probabilistically if a = f_1...f_r by another evaluation point
    # Step 27: Choose an evaluation point beta=b. 
    do 
        for i to j do 
            b[i] := rand(p)(); 
        end do;  
    
        # Step 28: Use dense interpolation to get an univariate image a(X[1],beta)
        a := Interp1var( B, X, b, deg, N, 1, p ); 

    until degree(a,X[1])=deg[1]; 

    Gcd(a,diff(a,X[1]),'Asf') mod p; 
    if singlepow > 0 then 
        Divide(Asf,X[1],'Asf') mod p; 
    end if;  
 
    if degree(Asf,X[1])<>Sdf1 then 
        userinfo(4, 'BBfactor', `Division check: Wrong degree of the square-free part of the interpolated univariate polynomial`); 
        return FAIL;  
    end if;

    fY := Array(1..r,rr->Eval(fN[rr],[seq(X[i]=b[i],i=2..j)]) mod p); 
    # Step 29: Check the correctness of lifted factors by divisions
    for rr to r do 
        if not Divide(Asf,fY[rr],'Asf') mod p then  
            userinfo(4, 'BBfactor', `Division check fails, possible failure of weak SHL assumption`); 
            return FAIL; 
        end if;
    end do;

    et := time() - st0; 
    if j = N then 
        userinfo( 5, 'BBfactor', `Total time for Hensel lifting step`,j,`is`,et ); 
    end if;

    return fN;
end:
