local MakeCont0, MakeCont, MakeBF, CMBBSHLcont;
# MakeCont0 makes a new black box of one less variable by evaluating the first variable at a fixed value alpha1. 
MakeCont0 := proc( B::procedure, # A modular black box for a polynomial f(x1,...,xn) in Z[x1,...,xn]. 
                   alpha1::integer, # evaluation point for x1, randomly chosen in Zp
                   LI::truefalse, # Indicator for large integer case
                   p::prime )::procedure; 
             # Output is a modular black box C (procedure) s.t. C(beta,p)=f(alpha1,beta_1,...,beta_{n-1}) mod p, where beta has size n-1. 
    return proc( beta::Array(integer), p::prime ) 
        local i::'integer', # loop index 
              nb::'integer':=numelems(beta), # nb = size of beta, which is n-1 
              alpha::'Array':=Array(1..nb+1,[alpha1,seq(beta[i],i=1..nb)],`if`(not LI,'datatype'='integer'[8],NULL));        
        return B( alpha, p ); #alpha = [alpha1,beta_1,...,beta_nb]
    end proc;
end proc;

# MakeBF makes a modular black box for a primitive polynomial f(x1,...,xn),i.e., 
# f has no content in x1 neither integer content, stored in its factored form.
MakeBF := proc( FAfac::Array(polynom), # Irreducible factors of f stored in an Array
                FApow::Array(integer), # Corresponding multiplicities of the factors stored in Array
                X::list(name), # list of variables
                p::prime )::procedure;
          # Ouput is a black box F (procedure) s.t. F( alpha, p ) = f(alpha) mod p.  
    # E.g. f = x1^10*(x2-x3)*(3*x1+x2^2-x3)^4*(x1+2*x2)^2. 
    # FAfac = Array(0..3, [x1, x2-x3, 3*x1+x2^2-x3, x1+2*x2])
    # FApow = Array(0..3, [10, 1, 4, 2]) 
    # FAfac and FApow is indexed from 0 to r, the number of irreducible factors which needed Hensel lifting.
    # FApow[0] stores the power of the first variable if exists (no Hensel lifting needed). Otherwise, it is 0.   
    local N := numelems(X), # Number of variables
          r := numelems(FAfac)-1; # Number of irreducible factors (without multiplicites) needed Hensel lifting
          return proc( alpha::Array(integer), p::prime )
              local tmult::'integer':=1, # Multiplied evaluations of the factors
              teval::'integer', # Evaluation of each factor
              i::'integer',  # loop index
              ii::'integer'; # loop index
	      for i from 0 to r do   
                  teval := Eval( FAfac[i], [seq(X[ii]=alpha[ii],ii=1..N)] ) mod p; 
                  teval := teval^FApow[i] mod p; 
                  tmult := tmult*teval mod p; 
              end do;
              return tmult;
          end proc;
end proc;

# MakeCont computes the quotient of two black boxes B and BF. 
# It is being used to compute the content of a polynomial f in the first variable by dividing out the evaluation of the primitive part of f.   
# The output is a procedure BC with one less variable s.t. BC(beta,p)=B(alpha,p)/BF(alpha,p), 
# where alpha1 is a fixed value, beta=[beta_1,...,beta_{n-1}], and alpha=[alpha1,beta_1,...,beta_{n-1}].
MakeCont := proc( B::procedure, 
                  BF::procedure, 
                  alpha1::integer, 
                  LI::truefalse, 
                  p::prime )::procedure; 
    return proc( beta::Array(integer), p::prime ) 
        local nb::'integer':=numelems(beta), # nb = size of beta, which is n-1
              alpha::'Array(integer)', # Evaluation point for f(x1,...,xn)
              g::'integer', # Evaluation BF(alpha, p);
              i::'integer'; # loop index
        alpha := Array(1..nb+1,[alpha1,seq(beta[i],i=1..nb)],`if`(not LI,'datatype'='integer'[8],NULL));
        g := BF( alpha, p ); 
        if g = 0 then 
            userinfo(4, 'BBfactor', `MakeCont failed: denominator evaluated to zero`); 
            return FAIL; 
        end if;
        return B( alpha, p )/g mod p; 
    end proc;
end proc; 


# CMBBSHLcont computes the factors of the input polynomial a(x1,...,xn) by recursively 
# computing the factors of the content one variable at a time. 
# E.g. let a = 21*(x4^3)*(2x3+x4+3)*(x2^2+x3+x4+3x5)*(x1^3+2*x1*x2+x3^3*x4+1)(x1+2)^2.
# The program first computes the factors of the primitive part of a, i.e. (x1^3+2*x1*x2+x3^3*x4+1)(x1+2)^2.
# Next, a new black is created for the content, 21*(x4^3)*(2x3+x4+3)*(x2^2+x3+x4+3x5). 
# Then, the factors of the primitive part of the content part is computed, we get (x2^2+x3+x4+3x5). 
# The process repeats until all factors of the content are computed. 

# Input: B, the modular black box of the input polynomial a(x1,...,xn),
#        X, the list of variables of a,
#        alpha, evaluation point chosen at random from Z_{Ntilde}^n, 
#        deg, partial degrees of a, computed before CMBBSHLcont,
#        tdeg, total degree of a, computed before CMBBSHLcont,
#        p, a prime, an optional user input in the BBfactor procedure.
#        VP, an indicator of whether the variables are permuted (true or false),
#        NoContNum (=N+1-primitive), a counter for recursive calls in CMBBSHLcont, 
#            NoContNum=N+1 by default, where primitive is the number of variables of the primitive factors to be computed. 
#        MapleCode = [false,false,false,false] by default, use C code for all 4 subroutines in CMBBSHL_stepj,
#        LI, an indicator for large integer case (true or false).
# Output: The irreducible factors of a(x1,...,xn) and their multiplicites
#         (with high probability correct). 
 
CMBBSHLcont := proc( B::procedure, # modular black box for a(x1,...,xn)
                     X::list(name), # list of variables 
                     alpha::Array(integer), # evaluation point
                     deg::list(integer), # partial degrees
                     tdeg::integer, # total degree
                     p::prime,  # main prime
                     NoContNum::nonnegint, # NoContNum=N+1-primitive, counter for recursive calls
                     MapleCode::table, # use C or Maple code for the 4 subroutines in CMBBSHL_stepj
                     $)::polynom;
               # Returns the product of the irreducible factors in expanded form, or FAIL or false.
               # FAIL indicates an unlucky evaluation point, false indicates to use a larger prime instead.  

    local N::'integer':=numelems(X), # number of variables
          LI::'truefalse':=`if`(p>2^61,true,false), # Boolean for large integer case, decided if p>2^61.
          F_pp::'polynom', # primitive factors
          CheckF_pp::'truefalse', # check answer for F_pp
          co::'polynom',  # content polynomial
          BBF::'procedure', # black box for the product of the primitive factors with multiplicities 
          BBC::'procedure', # black box for the content
          alphaC::'Array(integer)', # random evaluation point for the content
          j::'integer', # loop variable
          a1::'polynom',  # interpolated univariate polynomial a(x1,alpha_2,...,alpha_n)
          a1fac::'Array(polynom)', # stores the irreducible factors
          a1pow::'Array(integer)', # stores the multiplicites of the factors
          a1pow2::'Array(integer)', # stores the multiplicites of the factors
          alphaNew::'Array(integer)', # Array of evaluation points
          r::'integer', # number of irreducible factors of a, without multiplicities
          f1::'Array(polynom)', # Array of univariate factors
          fN::{'Array(polynom)',symbol}, # factors of fpp, the primitive part of a, after iratrecon
          fNA::'Array(polynom)', # Array of factors of n variables after Hensel lifting
          degC::'list', # partial degrees of the content in x1
          et::'float', # timer
          st::'float':=time(); # timer
    
    if N = 0 then
        # If no variables left, perform a Chinese remaindering to get the integer content.  
        if NoContNum = N+1 then # Only the primitive part is computed, integer content = 1. 
            F_pp := 1; 
            et := time()-st;
            userinfo(5, 'BBfactor', `total time for CMBBSHLcont at N=0:`, et); 
            return F_pp; 
        else 
           
            co := B_chrem( B, alpha, p ); # Chinese remaindering is done using several primes greater than p.  
            et := time()-st; 
            userinfo(5, 'BBfactor', `total time for CMBBSHLcont at N=1:`, et); 
            return co;
        end if;  

    elif N = 1 then
        # If only 1 variable left, densely interpolate a(x1,alpha_2,...,alpha_n) by Chinese remaindering with several primes greater than p. 
        a1 := Interp1varZ_chrem( B, X, alpha, deg, N, 1, p ); 

        # Check if there is an unlucky evaluation so the leading term vanishes.
        if degree(a1,X[1]) <> deg[1] then  
            userinfo(4, 'BBfactor', `unlucky evaluation at interpolating`, a1, `:wrong degree in`, X[1]); 
            return FAIL;
        end if; 
        
        a1 := factor(a1); # factor a(x1,alpha_2,...,alpha_n) in Z[x1]. 

        if NoContNum = N then   
            F_pp := a1/icontent(a1); # Only the primitive factors are computed, we remove the integer content.   
            et := time()-st; 
            userinfo(5, 'BBfactor', `total time for CMBBSHLcont at N=1:`, et); 
            return F_pp; 
      
        else  
            et := time()-st; 
            userinfo(5, 'BBfactor', `total time for CMBBSHLcont at N=1:`, et); 
            return a1; 
        end if;

    elif deg[1] = 0 then 
        # If the degree in the first variable is 0, we make a new black box (by calling MakeCont0)
        # for one less variable and pass it to the next level of recursion.
        
        if NoContNum = N then  
            return 1; # The primitive part is 1, if deg[1]=0. 
        end if;

        BBC := MakeCont0( B, rand(p)(), LI, p );  # Make a new black box for the content in x1.
        alphaC := ArrayTools:-Alias(alpha,1,[1..N-1]); # alphaC is an array = [alpha[2],...,alpha[N]]     
 
        # Recursively factor the content using the black box BBC.
        return CMBBSHLcont( BBC, X[2..N], alphaC, deg[2..N], tdeg, p, NoContNum, MapleCode );
 
    else 
        a1 := Interp1varZ_chrem( B, X, alpha, deg, N, 1, p ); # Densely interpolate a(x1,alpha_2,...,alpha_n) by Chinese remaindering  
        # check for unlucky evaluation in the leading coefficient, to see if the leading term vanished.
        if degree(a1,X[1]) <> deg[1] then 
            userinfo(4, 'BBfactor', `unlucky evaluation at interpolating`, a1,`:wrong degree in`,X[1]); 
            return FAIL;
        end if;

        # Store the factors of a1 into two Arrays: a1fac (the irreducible factors) and a1pow (their multiplicites).
        # E.g. a1 = 3*x1^5*(x1-2)^3*(2*x1^2-5)^2
        # a1fac := Array(0..2,[x1,x1-2,2*x1^2-5]);    
        # a1pow := Array(0..2,[5,3,2]); # Any power of the first variable is stored in a1pow[0].
        a1fac,a1pow := splitfactors(convert(factor(a1),'multiset')); 
        
        # Double check (with a different evaluation point, alphaNew) if a power of the first variable x1^dd (for some integer dd) exists 
        # (no need to Hensel lift) in the polynomial (with high probability), and it is indeed the same power as the first computation.
        if a1pow[0] <> 0 then 
            alphaNew := Array(1..N,i->rand(4001)(),`if`(not LI,'datatype'='integer'[8],NULL));
            a1 := Interp1varZ_chrem( B, X, alphaNew, deg, N, 1, p ); 
            a1pow2:= splitfactors(convert(factor(a1),'multiset'))[2]; 
           
            if a1pow2[0] = 0 or a1pow2[0] <> a1pow[0] then 
                userinfo(4, 'BBfactor', `unlucky evaluation at factoring`,a1,`:wrong power of`,X[1]); 
                return FAIL;    
            end if;
        
        end if;

        r := numelems(a1fac)-1; 
        if r = 0 then # The only factor is a power of x1. 
            fNA := a1fac;
	
        else
            f1 := map(modp,a1fac,p);  
            
            # --- Compute the primitive factors of a in x1 ---
            fN := CMBBSHL(B, f1[1..r], N, X, alpha, deg, p, a1pow[0], MapleCode );  
            # fN is an Array of the square-free and primitive factors after Hensel lifting. 
 
            if fN = FAIL or fN = false then 
                return fN; 
            end if;
            
            fN := iratrecon(fN, p); # This returns FAIL if any iratrecon(fN[i],p) for i=1..r returns a FAIL.  
             
            # If iratrecon returns FAIL, return false and use a larger prime
            if fN = FAIL then 
                userinfo(4, 'BBfactor', `iratrecon failed`); 
                return false; 
            end if;
            
            # Recover the coefficients of the factors from fractions to integers  
	    fN := map( numer, fN ); 
            
            # Store the recovered factors into an Array fNA.
	    fNA := ArrayTools:-Insert( fN, 0, X[1] );
                  
        end if; 
        
        # At this point, we have all the irreducible factors of the primitive part of a. 
        # We assume w.h.p. the multiplicites are the same as the univariate polynomial a1.
        # pack_nocontent takes inputs an Array of factors and an Array of their multiplicties and outputs a polynomial in factored form.
	F_pp := pack_nocontent( fNA, a1pow ); 
            
        # check_ans procedure checks the the computed factors probabilistically by using another prime q. 
        # It checks F_pp by interpolating fz(z) = F_pp(beta_1*z,...,beta_n*z) mod q and az(z) = a(beta_1*z,...,beta_n*z) mod q, 
        # and then dividing az(z) by fz(z) mod q. The last argument of check_ans is allfactors = false. 
        # If check_ans returns a false, it is likley caused by a missing term due to an unlucky prime. 
        # This program then returns a false, to use a larger prime in the BBfactor procedure.
        CheckF_pp := check_ans( B, F_pp, X, tdeg, prevprime(p), false ); 
        
        if CheckF_pp = false then
            userinfo(4, 'BBfactor', `After calculating F_pp: check_ans for F_pp returned false`); 
            return false; 
        end if;

        et := time()-st; 
        userinfo(5, 'BBfactor', `Total time for CMBBSHLcont at N = `,N,`is`,et); 
      
        if NoContNum = N then # Only want the primitive factors to be computed 
            return F_pp; 
        end if;
        
        # Make a black box for the product of the primitive factors (with multiplicities) 
        BBF := MakeBF( fNA, a1pow, X, p ); 
       
        # Make a new black box for the content of a(x1,...,xn)
        BBC := MakeCont( B, BBF, rand(p)(), LI, p );
        
        # --- Compute the primitive factors of the content --- 
        degC := deg - [seq(degree(F_pp,X[j]),j=1..N)]; # partial degrees of the content 
	alphaC := ArrayTools:-Alias(alpha,1,[1..N-1]); # alphaC is an array = [alpha[2],...,alpha[N]]     
        
        # Compute the factors of the content recursively.
        co := CMBBSHLcont( BBC, X[2..N], alphaC, degC[2..N], tdeg, p, NoContNum, MapleCode );
         
        if co = FAIL or co = false then 
            return co;
        end if;
 
        return co*F_pp; 

    end if;

end proc;
