local Interp2var_Maple;
# Interp2var_Maple interpolates a(beta[1],...,xi,...,xj,...,beta[N]) mod p
# using bivariate dense interpolation
# Input: A modular black box B that represents a(x1,..,xN)
#        X = [x1,...,xN], a list of variables, 
#        beta in Z^N is an Array of evaluation points,
#        deg is the list of partial degrees of all variables,
#        var = [vi,vj], two integers of variable indices, 1<=xi<xj<=N, 
#        p is a prime
# Ouput: a(beta[1],..,xi,..,xj,..,beta[N]) mod p
Interp2var_Maple := proc( B::procedure, 
                          X::list, 
                          beta::Array, 
                          var::list, 
                          deg::list, 
                          p::prime, 
                          singlepow::nonnegint, 
                          LI::truefalse ) 
    local vi::'integer':=var[1], # index number of xi 
          vj::'integer':=var[2], # index number of xj
          di::'integer':=deg[vi], # degree(a,xi)
          dj::'integer':=deg[vj], # degree(a,xj)
          xxi::'Array(integer)',   # points for interpolating xi
          xxj::'Array(integer)',  # points for interpolating xj
          i::'integer',  # loop variable 
          j::'integer',  # loop variable 
          k::'integer',  # loop variable 
          alpha := Array(beta,`if`(not LI, 'datatype'='integer'[8], NULL)), # evaluation points 
          F::'Array(integer)', # values for interpolating the variable xj
          g::'Array(polynom)', # interpolated polynomials in xj
          G::'Array(integer)', # Stores the coefficients of g[i] 
          sigma::'Array(polynom)'; # interpolated polynomials in xi
    xxi := Array(1..di+1,i->i); 
    xxj := Array(1..dj+1,i->i); 
    F := Array(1..dj+1);
    (g,G) := seq(Array(1..di+1),i=1..2);
    if singlepow > 0 then 
        for i to di+1 do 
            alpha[vi] := i;
            for j to dj+1 do 
                alpha[vj] := j; 
                F[j] := B( alpha, p )/alpha[vi]^singlepow mod p; 
            end do;
            g[i] := Interp( xxj, F, X[vj] ) mod p;
        end do;
    else 
        for i to di+1 do 
            alpha[vi] := i;
            for j to dj+1 do 
                alpha[vj] := j; 
                F[j] := B( alpha, p ); 
            end do;
            g[i] := Interp( xxj, F, X[vj] ) mod p;
        end do;
    end if;
    sigma := Array(0..dj);
    for j from 0 to dj do
        for i to di+1 do 
            G[i] := coeff(g[i],X[vj],j); 
        end do;
        sigma[j] := Interp( xxi, G, X[vi] ) mod p;
    end do;
    return Expand(add(sigma[k]*X[vj]^k, k=0..dj)) mod p; 
end proc:

