local GE;

# GE computes the determinant of A mod p using Gaussian elimination
GE:= proc( A::Matrix,
           p::prime )
    local n::'integer', # number of rows in A 
          m::'integer', # number of cols in A
          inv::'integer', # inverse of the diagonals
          mu::'integer', # intermediate variable
          i::'integer', # loop variable
          j::'integer', # loop variable
          k::'integer', # loop variable
          det::'integer'; # final answer
    n,m := op(1,A);
    if n<>m then 
        error "matrix must be square" 
    end if;
    det := 1;
    for k to n do
        i := k;
        while i<=n and A[i,k]=0 do 
            i := i+1; 
        end do;
        if i>n then 
            return 0; 
        end if;
        if i>k then # interchange row i and k
            for j from k to m do 
                A[i,j],A[k,j] := A[k,j],A[i,j] 
            end do;
            det := -det;
        end if;
        det := det*A[k,k] mod p;
        inv := 1/A[k,k] mod p;
        for i from k+1 to n do
            if A[i,k]=0 then 
                next 
            end if;
            mu := A[i,k]*inv mod p;
            for j from k+1 to n do
                A[i,j] := A[i,j]-mu*A[k,j] mod p;
            end do;
            A[i,k] := 0;
        end do;
    end do;
    return det;
end:

# MakeBlackBoxDet makes a black box B(alpha,p) = det(A) mod p
MakeBlackBoxDet := proc( A::Matrix, 
                         {VarPerm::list:=[]},
                         $ )::procedure;
    # Input: Matrix A with multivariate polynomial entries in x1,...,xN.
    #        VarPerm is a list of permutation of the variables x1,...,xN for optimization.
    #        E.g. VarPerm = [1,4,2,3] for X=[u,v,w,x], Xnew becomes [u,x,v,w].
    # Ouput: Black box BB(alpha,p) = det(A)(alpha) mod p,
    #        Xnew is a list of variables with the chosen permutation, which is the input 
    #        to BBfactor. 
    local n::'integer',  # number of rows of A 
          m::'integer',  # number of cols of A
          X::'list':=convert(indets(A),list), # list of variables
          N::'integer', # number of variables
          i::'integer', # loop variable
          Xnew::'list', # list of permuted variables  
          AL::'list':=convert(A,list), # A converted to a list
          AA::'Array',
          BB::'procedure';
    n,m := op(1,A);
    if n<>m then 
        error "matrix must be square" 
    end if;

    N := numelems(X);

    if VarPerm <> [] then
        if sort(VarPerm) <> [seq(i,i=1..N)] then 
            error "Input permutation incorrect: %1", VarPerm; 
        end if; 
        Xnew := [seq(X[VarPerm[i]],i=1..N)]; # list of permuted variables
    else 
        Xnew := X;
    end if;  

    AA := Array(1..n,1..n,'datatype'='integer'[8],'order'='C_order');
    BB := proc( alpha::Array, p::prime ) 
        global CNT,tBBeval,tBBdet; 
        local st::'float', # timer
              et::'float', # timer
              A_eval::'integer', # A evaluted at alpha
              d::'integer'; # ans = det(A)(alpha)
        CNT++; 
        if p < 2^61 then # Use C code for evaluation and determinant compt. 
            st := time(); 
            EVALMOD1( AL, Xnew, alpha, AA, p ); 
            et := time()-st; 
            tBBeval += et;
            
            st := time(); 
            d := Det64s(AA, n, p); 
            et := time()-st; 
            tBBdet += et;
        
        else # Use Maple code for evaluation and determinant compt.
            st := time(); 
            A_eval := Eval( A,[seq(Xnew[i]=alpha[i],i=1..N)]) mod p;
            et := time()-st; 
            tBBeval += et;
            
            st := time();
            d := GE(A_eval,p); 
            et := time()-st; 
            tBBeval += et;
        end if;    
        return d;
    end;
    return BB, Xnew;
end:

