# BBfactor procedure computes the irreducible factors of a multivariate polynomial given by a modular black box. 
# Inputs: 1. A modular black box B(alpha,p) = a(alpha) mod p, 
#            where a is a multivariate polynomial in Z[x1,...,xn], 
#            alpha = [alpha_1,...,alpha_n] in Z^n is an evaluation point.
#         2. A set of variables X, with any ordering chosen by the user, e.g. X=[x2,x1,x3,x5,x4]. 
#         Optional arguments:
#            maplecode chooses Maple or C code for each of the 4 subroutines in CMBBSHL_stepj.
#                The 4 subroutines are: 
#                1) evalfac: Evaluating the factors in the (j-1)th (j>2) Hensel lifting step;
#                2) interp2var: Interpolating the square-free part of the bivariate images of the polynomial, sqf(a(x1,xj));
#                3) BHL: Bivariate Hensel lifts;
#                4) VS: Solving Vandermonde systmes of equations.  
#                maplecode = [false,false,false,false] by default. 
#            primitive is the number of variables of the primitive factors to be computed.  
#                E.g. if primitive=0 (default), all factors of the content are to be computed. 
#                If primitive=1, the program only computes the primitive factors in x1. 
#                If primitive=2, only the primitive factors in x1, x2 are computed.
#                If primitive=n, primitive factors in x1,...,xn are computed, i.e. no integer content.
#            modulus is a large prime (the main prime), chosen by the user. 
#            bound is an upper bound for all the evaluation points s.t. bound < modulus.           
# Output: Irreducible factors of a(x1,...,xn) with high probability.  

BBfactor := proc( B::procedure, 
                X::list(name),  
               {maplecode::table:=table([evalfac=false,interp2var=false,BHL=false,VS=false])},
               {primitive::nonnegint:=0}, 
               {modulus::prime:=2305843009213693951}, 
               {bound::posint:=4001},
               $)::list([polynom,nonnegint]);
    global CNT,tBBeval,tBBdet,evalfac,interp2var,BHL,VS; 
    local N::'integer':=numelems(X), # Number of variables in X
          LI::'truefalse':=`if`(modulus>2^61,true,false), # Boolean for large integer case, decided if p>2^61.
          MC::'table':=maplecode, # Boolean
          allfactors::'truefalse':=`if`(primitive=0,true,false), # Boolean for if all factors of the content are computed.         
          j::'integer', # loop variable
          i::'integer', # loop variable
          p::'prime':=modulus, # main prime = 2^61-1 by default
          alpha::'Array', # Evaluation point in Z^n
          Ntilde::'integer':=bound, # A bound for the evaluation points in alpha, Ntilde=4001 by default. 
          deg::'list', # list of partial degrees
          tdeg::'integer', # total degree
          ff::'polynom', # factored polynomial or FAIL or false
          check::'truefalse', # answer for check_ans
          st::'float', # timer
          et::'float', # timer
          iter::'integer', # number of iterations to suceed the program 
          Xnew::'list(name)':=sort(X), # Variable names need to be sorted for the C subroutines
          final_ans::'list'; # list of final factors
    # For large integer case, use Maple code for all four major subroutines in CMBBSHL_stepj.
    if LI then 
        MC := table(['evalfac'=true,'interp2var'=true,'BHL'=true,'VS'=true]); 
    end if;
    if primitive > N then 
        error "primitive should be less than or equal to the number of variables"; 
    end if; 
    
    # Choice of Ntilde(bound) is in [N+1, p-1]. 
    if Ntilde < N + 1 then 
        error "bound is too small, must be greater than the number of variables"; 
    
    elif Ntilde > p - 1 then 
        Ntilde := min(4001, p - 1);
        WARNING("bound is too big, reset to %1",Ntilde);
    end if; 
 
    # The main loop  
    for iter to 20 do # upper bound for iter = 20
       
        # --- Select alpha's at random --- 
        alpha := Array(1..N,i->rand(Ntilde)(),`if`(not LI,'datatype'='integer'[8], NULL));
        userinfo(3, 'BBfactor', `large integer case?`, LI); 
        userinfo(3, 'BBfactor', `modulus and bound:`, p, Ntilde);
   
        # ---- Compute deg(a,xj) w.h.p. and the total degree w.h.p. ---- 
        (CNT, tBBeval, tBBdet) := 0, 0, 0;
        st := time():
        deg := [seq(degB_nobound(B,N,j,p,LI),j=1..N)]; # Partial degrees 
        tdeg := TotalDeg( B, N, p, LI ); # Total degree
        et := time()-st: 
        
        userinfo(3, 'BBfactor', `degrees of the input polynomial:`, deg);
        userinfo(4, 'BBfactor', `total number of probes for degree computation:`, CNT); 
        userinfo(4, 'BBfactor', `total time for degree computation:`, et);
        userinfo(5, 'BBfactor', `total times for deg compt: BBeval and BBdet:`, tBBeval, tBBdet);
 
        # ---- Perform factorization ----  
        (CNT, tBBeval, tBBdet) := 0, 0, 0;
        st := time():
        ff := CMBBSHLcont( B, Xnew, alpha, deg, tdeg, p, N+1-primitive, MC );
        et := time() - st:
        
        userinfo(4, 'BBfactor', `total number of probes for CMBBSHLcont:`, CNT);
        userinfo(4, 'BBfactor', `total time for CMBBSHLcont:`, et);
    
        if ff = FAIL then 
            # FAIL indicates an unlucky evaluation. Pick a previous prime and try again. Ntilde is the same.  
            p := prevprime(p); 
            next; 

        elif ff = false then # false indicates to use a larger prime next.  
            p := nextprime(p^2);

            if p > 2^61 then 
                LI := true; 
                MC := table(['evalfac'=true,'interp2var'=true,'BHL'=true,'VS'=true]); 
            end if;

            Ntilde := Ntilde^2;  
            next; 
        end if; 
    
        # ---- Check answer by choosing a previous prime ---- 
        # The program did not return FAIL (or false) previously, and check_ans returns false indicates 
        # there is likely an unlucky prime to cause a missing term, so we enlarge the prime directly.
        check := check_ans( B, ff, Xnew, tdeg, prevprime(p), allfactors );  
        if check = false then 
            p := nextprime(p^2); 
            
            if p > 2^61 then 
                LI := true; 
                MC := table(['evalfac'=true,'interp2var'=true,'BHL'=true,'VS'=true]);
            end if;
           
            next; 
        end if;   
         
    until check = true;
    
    if check <> true then 
        userinfo(3, 'BBfactor', `number of iterations to increase the prime exceeded the upper bound`); 
        return FAIL;  
    else     
        userinfo(4, 'BBfactor', `total number of primes used to succeed the program:`,iter);         
        final_ans := convert(ff, 'multiset'); # returns a list of the factors with their multiplicites
       
        # Returns the factors in the same order of variables as input X
        if X <> Xnew then 
            return subs( [seq(Xnew[i]=X[i],i=1..N)],final_ans );
        else 
            return final_ans;
        end if;
 
    end if;

end proc: 
