local MakeBBu;
 
# Make a univariate black box BBu from a multivariate black box BB for a(x1,...,xn) s.t
# BBu(z,q) = a( beta_1*z,...,beta_n*z ) mod q.
MakeBBu := proc( BB::procedure, 
                 beta::Array(integer), 
                 LI::truefalse )
    return proc( aa::Array(integer), pp::prime ) 
        # Input BBu(aa,pp): aa has only one element, pp is a prime.
        local N::'integer':=numelems(beta), # size of beta
              aA::'Array';
        aA := Array(1..N,i->aa[1]*beta[i] mod pp,`if`(not LI,'datatype'='integer'[8],NULL));
        return BB( aA, pp );
    end;
end proc;

# check_ans procedure checks the computed factors probabilistically by using another prime q.
# It is used for checking the computed primitive factors of a(x1,...,xn). 
# Let fpp(x1,...,xn) be a computed product of the primitive factors of a(x1,...,xn) with their multiplicites.
# The algorithm in check_ans is the following:
# 1. Pick a random evaluation point beta = [beta_1,...,beta_n] in Z_q^n.
# 2. Interpolate az(z):=a(x1=beta_1*z, ..., xn=beta_n*z) mod q by a univariate dense interpolation.  
# 3. Evaluate fz(z):=fpp(beta_1*z,...,beta_n*z) mod q.   
# 4. Divide az(z) by fz(z) mod q. If the remainder is 0, the factors are correct with high probability. 

check_ans := proc( B::procedure, # The modular black box for a(x1,...,xn)
                   f::polynom,   # A product of computed factors of a with their multiplicities.  
                   X::list(name), # list of variables x1,...,xn
                   tdeg::integer, # bound of total degree of a
                   q::prime,      # a previous prime of the main prime used in CMBBSHLcont
                   allfactors::truefalse # whether all factors have been computed 
                   )::truefalse; # Output is true or false. 
    local N::'integer':=numelems(X), # number of variables in X 
          LI::'truefalse':=`if`(q>2^61,true,false), # Boolean for large integer case, decided if q>2^61.
          betaA::'Array', # betaA=[beta_1,...,beta_n], random evaluation point in Z_q^n
          ii::'integer', # loop variable
          f_eval::'integer', # Evaluation of f at betaA 
          BBu::'procedure',  # Univariate modular black box for az(z)=a(beta_1*z,...,beta_n*z) 
          az::'polynom', # interpolated polynomial az(z) = a(beta_1*z,...,beta_n*z) 
          fz::'polynom', # interpolated f(z) = f(beta_1*z,...,beta_n*z)
          z::'symbol'; # variable name
    # Step 1:
    betaA := Array(1..N,i->rand(1..q)(),`if`(not LI, 'datatype'='integer'[8],NULL)); 
    
    if allfactors then 
        # All factors have been computed. Check answer by evaluating f(betaA) mod q and B(betaA,q), 
        f_eval := Eval(f, [seq(X[ii]=betaA[ii],ii=1..N)]) mod q; # Step 3
        # Step 4 (modified for allfactors=true):
        if f_eval <> B(betaA,q) then 
            userinfo(4, 'check_ans', `check answer: all factors of the content computed: incorrect`); 
            return false; 

        else 
            userinfo(4, 'check_ans', `check answer: all factors of the content computed: correct`); 
            return true; 
        end if; 

    else     
        # Make a univariate black box BBunivar
        BBu := MakeBBu( B, betaA, LI );
        az := Interp1var( BBu, [z], Array([1]), [tdeg], 1, 1, q );  # Step 2   
        fz := Eval( f, [seq(X[ii]=betaA[ii]*z,ii=1..N)] ) mod q; # Step 3
        
        # Step 4: 
        if Rem( az, fz, z ) mod q = 0 then 
            userinfo(4, 'check_ans', `check answer: not all factors of the content computed: correct`); 
            return true; 
      
        else 
            userinfo(4, 'check_ans', `check answer: not all factors of the content computed: incorrect`); 
            return false; 
        end if;
    
    end if;

end:
 
