local ConstructU, UnivDeg, TotalDeg;
# TotalDeg computes the total degree of a polynomial represented by a black box B.
# TotalDeg calls ConstructU and UnivDeg.

ConstructU := proc( B::procedure, 
                    beta::Array, 
                    p::prime, 
                    LI::truefalse )
    # Input: A multivariate black box B s.t. B(beta,p) = a(beta) mod p, 
    #        beta = [beta_1,...,beta_N], an evaluation point, 
    #        p is a large prime, 
    #        LI = true or false, for large integer case.
    # Output: A univariate black box U s.t. 
    #         U(alpha,p) = B([beta_1*alpha,...,beta_N*alpha],p)
    # Ref: Algorithm 1 in PhD thesis (Chen 2024) 
    local N::'integer':=numelems(beta); 
    proc( alpha::integer, p::prime ) 
        local beta2::'Array':=Array(1..N,i->beta[i]*alpha mod p, `if`(not LI,'datatype'='integer'[8],NULL));
        return B( beta2, p ); 
    end;
end proc;

UnivDeg := proc( U::procedure, 
                 p::prime )
    # Computing the degree of f(x) in Z[x] w.h.p.
    # Input: A univariate black box U s.t. U(alpha) = f(alpha) mod p, p is a prime
    # Output: deg(f) w.h.p.
    # Reference: Algorithm 2 in PhD thesis (Chen 2024)
    local randzp::'procedure':=rand(p),
          g::'table',
          k::'integer', # loop variable, final degree
          m::'polynom',
          x::'name',
          alpha::'table',
          m_eval::'integer',
          y::'table',
          v::'table';
    g[-1] := 0; 
    (k,m) := 0,1; 
    while true do 
        do 
            alpha[k] := randzp(); 
            m_eval := Eval(m,x=alpha[k]) mod p; 
        until m_eval <> 0;  
        y[k] := U( alpha[k], p );
        v[k] := (y[k] - Eval(g[k-1],x=alpha[k]) mod p)/m_eval mod p; 
        if v[k] = 0 then 
            return k-1; 
        end if;
        g[k] := g[k-1] + v[k]*m; 
        m := m*(x-alpha[k]);
        k++;
     end do;
end proc;

TotalDeg := proc( B::procedure, 
                  N::nonnegint, 
                  p::prime, 
                  LI::truefalse )
    # Computing the total degree of a in Z[x1,...,xn] w.h.p.
    # Input: A multivariate black box B s.t. B(alpha,p) = a(alpha) mod p,
    #        N is the number of variables,
    #        p is a large prime,
    #        LI = true or false for large integer case 
    # Output: deg(a) w.h.p.
    # Ref: Algorithm 3 in PhD thesis (Chen 2024)
    local U::'procedure',
          randzp::'procedure':=rand(p), 
          beta::'Array':=Array(1..N,i->randzp(),`if`(not LI, 'datatype'='integer'[8],NULL));
    U := ConstructU( B, beta, p, LI );
    return UnivDeg( U, p );
end proc;
